const { parse } = require("dotenv");
const { reputationmanagementTemplate , reputationmanagementTemplateModel  } = require("../../models/services_templates/reputationmanagementTemplateModel");
require("dotenv").config();
const reviews_faq_filterController = require("./reviews_faqs_filter/reviews_faq_filterController");




const withBaseUrl = (path) => {
  if (!path) return path;
  if (path.startsWith("http")) return path;
  return `${BASE_URL}${path}`;
};

const parseSafe = (json) => {
  try {
    return typeof json === 'string' ? JSON.parse(json) : json;
  } catch {
    return null;
  }
};

const BASE_URL = process.env.BASE_URL;

const reputationmanagementTemplateController = {
  createTemplate: async (req, res) => {
    try {
      const newData = await reputationmanagementTemplateModel.create(req.body);
      return res.status(201).json({
        success: true,
        message: "Virtual Employees Template created!",
        data: newData,
      });
    } catch (error) {
      console.error("Create Error:", error);
      return res.status(500).json({ success: false, message: "Server Error", error });
    }
  },

  getAllTemplates: async (req, res) => {
  try {
    const templates = await reputationmanagementTemplateModel.findAll({
      order: [["created_at", "DESC"]],
    });

    const parsedTemplates = templates.map(template => {
      const t = template.toJSON();

      const banner = parseSafe(t.banner);
      if (banner?.background_image_url) {
        banner.background_image_url = withBaseUrl(banner.background_image_url);
      }

      const standOut = parseSafe(t.stand_out_section);
      if (standOut?.background_image_url) {
        standOut.background_image_url = withBaseUrl(standOut.background_image_url);
      }

      const ourAchievements = parseSafe(t.our_achievements_section);
      if (ourAchievements?.features) {
        ourAchievements.features = ourAchievements.features.map(f => ({
          ...f,
          icon_url: withBaseUrl(f.icon_url),
        }));
      }

      const ourExperties = parseSafe(t.our_experties_section);
      if (ourExperties?.experties_cards) {
        ourExperties.experties_cards = ourExperties.experties_cards.map(card => ({
          ...card,
          info_cards: card.info_cards?.map(info => ({
            ...info,
            image_url: withBaseUrl(info.image_url),
          })) || []
        }));
      }

      const portfolio = parseSafe(t.portfolio_section);
      if (portfolio?.sections_cards) {
        portfolio.sections_cards = portfolio.sections_cards.map(c => ({
          ...c,
          icon_url: withBaseUrl(c.icon_url),
        }));
      }

      return {
        ...t,
        banner,
        stand_out_section: standOut,
        our_achievements_section: ourAchievements,
        our_experties_section: ourExperties,
        portfolio_section: portfolio,
      };
    });

    return res.status(200).json({ success: true, data: parsedTemplates });
  } catch (error) {
    console.error("Fetch Error:", error);
    return res.status(500).json({ success: false, message: "Server Error", error });
  }
},

getTemplateById: async (req, res) => {
  try {
    const id = req.params.id;
    const template = await reputationmanagementTemplateModel.findByPk(id);
    if (!template) {
      return res.status(404).json({ success: false, message: "Template not found" });
    }

    const t = template.toJSON();

    const banner = parseSafe(t.banner);
    if (banner?.background_image_url) {
      banner.background_image_url = withBaseUrl(banner.background_image_url);
    }

    const standOut = parseSafe(t.stand_out_section);
    if (standOut?.background_image_url) {
      standOut.background_image_url = withBaseUrl(standOut.background_image_url);
    }

    const ourAchievements = parseSafe(t.our_achievements_section);
    if (ourAchievements?.features) {
      ourAchievements.features = ourAchievements.features.map(f => ({
        ...f,
        icon_url: withBaseUrl(f.icon_url),
      }));
    }

    const ourExperties = parseSafe(t.our_experties_section);
    if (ourExperties?.experties_cards) {
      ourExperties.experties_cards = ourExperties.experties_cards.map(card => ({
        ...card,
        info_cards: card.info_cards?.map(info => ({
          ...info,
          image_url: withBaseUrl(info.image_url),
        })) || []
      }));
    }

    const portfolio = parseSafe(t.portfolio_section);
    if (portfolio?.sections_cards) {
      portfolio.sections_cards = portfolio.sections_cards.map(c => ({
        ...c,
        icon_url: withBaseUrl(c.icon_url),
      }));
    }

      const get_reviews_url = `reputation-management-service-template/${id}`;
      const reviews = await reviews_faq_filterController.filterReviewsByRoute(get_reviews_url);

      const get_faqs = `reputation-management-service-template/${id}`;
      const faqs = await reviews_faq_filterController.getFaqForPage(get_faqs);

    return res.status(200).json({
      success: true,
      data: {
        ...t,
        banner,
        stand_out_section: standOut,
        our_achievements_section: ourAchievements,
        our_experties_section: ourExperties,
        portfolio_section: portfolio,
        reviews: reviews,
        faqs: faqs
      },
    });
  } catch (error) {
    console.error("Get By ID Error:", error);
    return res.status(500).json({ success: false, message: "Server Error", error });
  }
},



  updateTemplate: async (req, res) => {
    try {
      const id = req.params.id;
      const existing = await reputationmanagementTemplateModel.findByPk(id);
      if (!existing) {
        return res.status(404).json({ success: false, message: "Template not found" });
      }

      await reputationmanagementTemplateModel.update(req.body, { where: { id } });
      const updated = await reputationmanagementTemplateModel.findByPk(id);

      return res.status(200).json({
        success: true,
        message: "Virtual Employees Template updated!",
        data: updated,
      });
    } catch (error) {
      console.error("Update Error:", error);
      return res.status(500).json({ success: false, message: "Server Error", error });
    }
  },

  deleteTemplate: async (req, res) => {
    try {
      const id = req.params.id;
      const deleted = await reputationmanagementTemplateModel.destroy({ where: { id } });

      if (!deleted) {
        return res.status(404).json({ success: false, message: "Template not found" });
      }

      return res.status(200).json({
        success: true,
        message: "Template deleted successfully!",
      });
    } catch (error) {
      console.error("Delete Error:", error);
      return res.status(500).json({ success: false, message: "Server Error", error });
    }
  },
};

module.exports = reputationmanagementTemplateController;
